﻿using gov.va.med.vbecs.Common;
using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;

namespace gov.va.med.vbecs.BOL
{
    /// <summary>
    /// PendingTestHelper used to help parse pending test data for specimen and blood units
    /// </summary>
    public static class PendingTestHelper
    {
        /// <summary>
        /// NOT_TESTED_SYMBOL = X
        /// </summary>
        public const string NOT_TESTED_SYMBOL = "X";
        /// <summary>
        /// COMPATIBLE_SAFE_TO_TRANSFUSE_ID = C
        /// </summary>
        public const string COMPATIBLE_SAFE_TO_TRANSFUSE_ID = "C";
        /// <summary>
        /// COMPATIBLE_DONT_TRANSFUSE_ID = D
        /// </summary>
        public const string COMPATIBLE_DONT_TRANSFUSE_ID = "D";
        /// <summary>
        /// INCOMPATIBLE_ONLY_MEDICAL_DIRECTOR_ID = G
        /// </summary>
        public const string INCOMPATIBLE_ONLY_MEDICAL_DIRECTOR_ID = "G";
        /// <summary>
        /// INCOMPATIBLE = I
        /// </summary>
        public const string INCOMPATIBLE = "I";
        /// <summary>
        /// INTERP_NAME = "interp"
        /// </summary>
        public const string INTERP_NAME = "interp";

        /// <summary>
        /// GetBloodTestTypeValue
        /// </summary>
        /// <param name="testType"></param>
        /// <param name="pendingBloodUnitTestDaoList"></param>
        /// <returns></returns>
        public static PendingTestResultModel GetPendingTestTypeValue(TestType testType, IEnumerable<IPendingTestDao> pendingBloodUnitTestDaoList)
        {
            var pendingTest = pendingBloodUnitTestDaoList.FirstOrDefault(x => x.BloodTestTypeId == (int)testType);

            if (pendingTest == null)
            {
                return new PendingTestResultModel(0, testType, "", NOT_TESTED_SYMBOL, NOT_TESTED_SYMBOL, null);
            }
            else
            {
                var resultId = pendingTest.TestResultId.Trim();
                var resultText = pendingTest.TestResultText.Trim();

                if (resultId == "I")
                {
                    resultText = "Incompatible";
                }

                return new PendingTestResultModel(pendingTest.PendingTestId, testType, pendingTest.BloodTestTypeName.Trim(), resultId, resultText, pendingTest.RowVersion);
            }
        }

        /// <summary>
        /// AssignPropertiesFromDao used to assign base class properties
        /// </summary>
        /// <param name="pendingTestModel"></param>
        /// <param name="pendingTestDaoList"></param>
        public static void AssignPropertiesFromDao(IPendingTestModel pendingTestModel, IEnumerable<IPendingTestDao> pendingTestDaoList)
        {
            if (pendingTestDaoList != null &&
                pendingTestDaoList.Any())
            {
                // Sort so we loop thorugh the more important ones first
                pendingTestDaoList = pendingTestDaoList.OrderBy(x => x.BloodTestTypeSortOrder).ToList();
                var pendingBloodUnitTestDao = pendingTestDaoList.First();

                // Get the test comment of the first test that has a comment entered
                var autoInstrumentComments = pendingTestDaoList.Where(x => !string.IsNullOrWhiteSpace(x.AutoInstrumentComments)).Select(s => s.AutoInstrumentComments).FirstOrDefault();
                // Get the AutoInstrumentName of the first test that has a value for it
                var autoInstrumentName = pendingTestDaoList.Where(x => !string.IsNullOrWhiteSpace(x.AutoInstrumentName)).Select(s => s.AutoInstrumentName).FirstOrDefault();

                pendingTestModel.InstrumentUserId = pendingBloodUnitTestDao.TestTechId;
                pendingTestModel.InstrumentUserName = pendingBloodUnitTestDao.TestTechUserName;
                // Set the user name to the UserId if we couldn't find a match
                if (string.IsNullOrWhiteSpace(pendingTestModel.InstrumentUserName))
                {
                    pendingTestModel.InstrumentUserName = pendingTestModel.InstrumentUserId;
                }
                pendingTestModel.TestedDateTime = pendingBloodUnitTestDao.TestDate;
                pendingTestModel.AutoInstrumentComments = autoInstrumentComments;
                pendingTestModel.PendingTestStatus = (PendingTestStatus)pendingBloodUnitTestDao.PendingTestStatusId;
                pendingTestModel.OrderableTest = new OrderableTestModel(pendingBloodUnitTestDao.OrderableTestId, pendingBloodUnitTestDao.OrderableTestName);
                pendingTestModel.DivisionCode = pendingBloodUnitTestDao.DivisionCode;
                pendingTestModel.InstrumentName = autoInstrumentName;
                pendingTestModel.LastUpdateUser = pendingBloodUnitTestDao.LastUpdateUser;

                // Add Test Results
                foreach (var pendingTest in pendingTestDaoList)
                {
                    TestType bloodTestType;

                    if (Enum.TryParse(pendingTest.BloodTestTypeId.ToString(), out bloodTestType))
                    {
                        var testResult = PendingTestHelper.GetPendingTestTypeValue(bloodTestType, pendingTestDaoList);

                        if (testResult != null)
                        {
                            pendingTestModel.TestResultList.Add(testResult);
                        }
                    }
                }
            }
        }

        /// <summary>
        /// GetProductCode
        /// </summary>
        /// <param name="pendingBloodUnitDao"></param>
        /// <returns></returns>
        public static BloodUnitModel GetBloodUnitModel(IBloodUnitDao pendingBloodUnitDao)
        {
            // If test matches at least one BloodUnit try to set the ProductCode and AboRh values
            if (pendingBloodUnitDao != null)
            {
                return new BloodUnitModel(pendingBloodUnitDao.BloodUnitGuid,
                                          pendingBloodUnitDao.BloodUnitStatusGuid,
                                          pendingBloodUnitDao.UnitProductCode,
                                          pendingBloodUnitDao.DonationTypeCode,
                                          pendingBloodUnitDao.UnitDivisionId,
                                          pendingBloodUnitDao.ProductShortName,
                                          pendingBloodUnitDao.BloodTypeCode,
                                          pendingBloodUnitDao.RhFactorText,
                                          Utility.GetUnitStatusCodeFromString(pendingBloodUnitDao.UnitStatusCode),
                                          pendingBloodUnitDao.LabelTypeCode,
                                          pendingBloodUnitDao.BloodUnitStatusRowVersion);
            }

            return null;
        }

        /// <summary>
        /// Build Workload Table for Accepting
        /// Defect 251218
        /// </summary>
        /// <param name="pendingTestList"></param>
        /// <param name="updateFunction"></param>
        /// <param name="dtWorkLoad"></param>
        /// <param name="isWorkloadDefined"></param>
        public static void BuildWorkLoadTableForAccepting(IEnumerable<IPendingTestModel> pendingTestList, UpdateFunction updateFunction, ref DataTable dtWorkLoad, ref bool isWorkloadDefined)
        {
            isWorkloadDefined = false;
            dtWorkLoad = null;

            if (pendingTestList == null) return;
            bool foundNoWorkLoad = false;

            foreach (var pendingTest in pendingTestList)
            {
                // Workload
                DataTable dtWorkLoadProcessGuid = null;
                var workloadProcessId = pendingTest.WorkloadProcessId;
                dtWorkLoadProcessGuid = BOL.WorkloadEvent.GetWorkloadProcessGuidByProcessID((int)workloadProcessId);

                if (dtWorkLoadProcessGuid == null ||
                    dtWorkLoadProcessGuid.Rows.Count == 0)
                {
                    foundNoWorkLoad = true;
                }

                if (workloadProcessId != WorkloadProcessID.Unknown)
                {
                    DataTable dtTempWorkLoad = null;

                    if (pendingTest is IPendingBloodUnitTestModel)
                    {
                        var bloodUnitGuidArray = new Guid[] { ((IPendingBloodUnitTestModel)pendingTest).BloodUnitModel.BloodUnitGuid };
                        dtTempWorkLoad = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, bloodUnitGuidArray, Common.WorkloadTransactionType.Unit, false);
                    }
                    else if (pendingTest is IPendingSpecimenTestModel)
                    {
                        var patientGuidArray = new Guid[] { ((IPendingSpecimenTestModel)pendingTest).OrderedTest.PatientGuid };
                        dtTempWorkLoad = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, patientGuidArray, Common.WorkloadTransactionType.Patient, false);
                    }

                    if (dtTempWorkLoad != null)
                    {
                        if (dtWorkLoad == null)
                        {
                            dtWorkLoad = dtTempWorkLoad;
                        }
                        else
                        {
                            //*** Fortify Justified Code ***
                            //dtWorkLoad or dtTempWorkLoad can't be null since it is checked earler (see above). 
                            foreach (DataRow dtRow in dtTempWorkLoad.Rows)
                            {
                                DataRow newRow = dtWorkLoad.NewRow();
                                newRow.ItemArray = dtRow.ItemArray;
                                dtWorkLoad.Rows.Add(newRow);
                            }
                            //*** Fortify Justified Code ***
                        }
                    }
                }
            }

            if (dtWorkLoad != null)
            {
                Common.Utility.AppendLastUpdateInformation(dtWorkLoad, updateFunction);
            }

            isWorkloadDefined = !foundNoWorkLoad;
        }

        /// <summary>
        /// Determines if orderable test is for Antigen Typing (Looks for AG in enum name)
        /// </summary>
        /// <param name="orderableTest"></param>
        /// <returns></returns>
        public static bool IsAntigenTypingOrderableTest(OrderableTest orderableTest)
        {
            return (orderableTest.ToString().IndexOf("AG", StringComparison.InvariantCulture) >= 0);

            //    if (                    case OrderableTest.AGA1:
            //            case OrderableTest.AGc:
            //            case OrderableTest.AGC:
            //            case OrderableTest.AGCe:
            //            case OrderableTest.AGCw:
            //            case OrderableTest.AGCx:
            //            case OrderableTest.AGDia:
            //            case OrderableTest.AGDib:
            //            case OrderableTest.AGe:
            //            case OrderableTest.AGE:
            //            case OrderableTest.AGFya:
            //            case OrderableTest.AGFyb:
            //            case OrderableTest.AGi:
            //            case OrderableTest.AGI:
            //            case OrderableTest.AGJka:
            //            case OrderableTest.AGJkb:
            //            case OrderableTest.AGJsa:
            //            case OrderableTest.AGJsb:
            //            case OrderableTest.AGk:
            //            case OrderableTest.AGK:
            //            case OrderableTest.AGKpa:
            //            case OrderableTest.AGKpb:
            //            case OrderableTest.AGLea:
            //            case OrderableTest.AGLeb:
            //            case OrderableTest.AGLua:
            //            case OrderableTest.AGLub:
            //            case OrderableTest.AGLW:
            //            case OrderableTest.AGM:
            //            case OrderableTest.AGN:
            //            case OrderableTest.AGP:
            //            case OrderableTest.AGP1:
            //            case OrderableTest.AGPk:
            //            case OrderableTest.AGPP1:
            //            case OrderableTest.AGs:
            //            case OrderableTest.AGS:
            //            case OrderableTest.AGSc1:
            //            case OrderableTest.AGSc2:
            //            case OrderableTest.AGU:
            //            case OrderableTest.AGwD:
            //            case OrderableTest.AGXgA:
            //            case OrderableTest.AGYta:
            //            case OrderableTest.AGYtb:)

            //    RepeatAGwD = 61,
            ///// <summary>
            ///// RepeatAGC = 62
            ///// </summary>
            //RepeatAGC = 62,
            ///// <summary>
            ///// RepeatAGE = 63
            ///// </summary>
            //RepeatAGE = 63,
            ///// <summary>
            ///// RepeatAGc = 64
            ///// </summary>
            //RepeatAGc = 64,
            ///// <summary>
            ///// RepeatAGe = 65
            ///// </summary>
            //RepeatAGe = 65,
            ///// <summary>
            ///// RepeatAGK = 70
            ///// </summary>
            //RepeatAGK = 70,
            ///// <summary>
            ///// RepeatAGk = 71
            ///// </summary>
            //RepeatAGk = 71,
            ///// <summary>
            ///// RepeatAGKpa = 72
            ///// </summary>
            //RepeatAGKpa = 72,
            ///// <summary>
            ///// RepeatAGKpb = 73
            ///// </summary>
            //RepeatAGKpb = 73,
            ///// <summary>
            ///// RepeatAGJka = 74
            ///// </summary>
            //RepeatAGJka = 74,
            ///// <summary>
            ///// RepeatAGJkb = 75
            ///// </summary>
            //RepeatAGJkb = 75,
            ///// <summary>
            ///// RepeatAGFya = 78
            ///// </summary>
            //RepeatAGFya = 78,
            ///// <summary>
            ///// RepeatAGFyb = 79
            ///// </summary>
            //RepeatAGFyb = 79,
            ///// <summary>
            ///// RepeatAGM = 80
            ///// </summary>
            //RepeatAGM = 80,
            ///// <summary>
            ///// RepeatAGN = 81
            ///// </summary>
            //RepeatAGN = 81,
            ///// <summary>
            ///// RepeatAGS = 82
            ///// </summary>
            //RepeatAGS = 82,
            ///// <summary>
            ///// RepeatAGs = 83
            ///// </summary>
            //RepeatAGs = 83,
            ///// <summary>
            ///// RepeatAGU = 84
            ///// </summary>
            //RepeatAGU = 84,
            ///// <summary>
            ///// RepeatAGP1 = 85
            ///// </summary>
            //RepeatAGP1 = 85,
            ///// <summary>
            ///// RepeatAGPP1 = 86
            ///// </summary>
            //RepeatAGPP1 = 86,
            ///// <summary>
            ///// RepeatAGLea = 87
            ///// </summary>
            //RepeatAGLea = 87,
            ///// <summary>
            ///// RepeatAGLeb = 88
            ///// </summary>
            //RepeatAGLeb = 88,
            ///// <summary>
            ///// RepeatAGJsa = 89
            ///// </summary>
            //RepeatAGJsa = 89,
            ///// <summary>
            ///// RepeatAGJsb = 90
            ///// </summary>
            //RepeatAGJsb = 90,
            ///// <summary>
            ///// RepeatAGLua = 91
            ///// </summary>
            //RepeatAGLua = 91,
            ///// <summary>
            ///// RepeatAGLub = 92
            ///// </summary>
            //RepeatAGLub = 92,
            ///// <summary>
            ///// RepeatAGCw =  95
            ///// </summary>
            //RepeatAGCw =  95,
            ///// <summary>
            ///// RepeatAGCe = 96
            ///// </summary>
            //RepeatAGCe = 96,
            ///// <summary>
            ///// RepeatAGCx = 97
            ///// </summary>
            //RepeatAGCx = 97,
            ///// <summary>
            ///// RepeatAGXgA = 98
            ///// </summary>
            //RepeatAGXgA = 98,
            ///// <summary>
            ///// RepeatAGP = 99
            ///// </summary>
            //RepeatAGP = 99,
            ///// <summary>
            ///// RepeatAGPk = 100
            ///// </summary>
            //RepeatAGPk = 100,
            ///// <summary>
            ///// RepeatAGA1 = 101
            ///// </summary>
            //RepeatAGA1 = 101,
            ///// <summary>
            ///// RepeatAGI = 102
            ///// </summary>
            //RepeatAGI = 102,
            ///// <summary>
            ///// RepeatAGi = 103
            ///// </summary>
            //RepeatAGi = 103,
            ///// <summary>
            ///// RepeatAGDia = 104
            ///// </summary>
            //RepeatAGDia = 104,
            ///// <summary>
            ///// RepeatAGDib = 105
            ///// </summary>
            //RepeatAGDib = 105,
            ///// <summary>
            ///// RepeatAGLW = 106
            ///// </summary>
            //RepeatAGLW = 106,
            ///// <summary>
            ///// RepeatAGSc1 = 107
            ///// </summary>
            //RepeatAGSc1 = 107,
            ///// <summary>
            ///// RepeatAGSc2 = 108
            ///// </summary>
            //RepeatAGSc2 = 108,
            ///// <summary>
            ///// RepeatAGYta = 109
            ///// </summary>
            //RepeatAGYta = 109,
            ///// <summary>
            ///// RepeatAGYtb = 110
            ///// </summary>
            //RepeatAGYtb = 110,
        }
    }
}
